<?php

namespace App\Controllers;

use CodeIgniter\HTTP\ResponseInterface;

class ImageController extends BaseController
{
    public function serve($folder, $filename)
    {
        // Validate folder to prevent directory traversal
        $allowedFolders = ['biodata', 'activities', 'projects', 'certificates', 'training', 'work', 'organizations'];
        
        if (!in_array($folder, $allowedFolders)) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Folder not found');
        }
        
        // Build file path in assets/uploads
        $filePath = FCPATH . 'assets/uploads/' . $folder . '/' . $filename;
        
        // Check if file exists
        if (!file_exists($filePath)) {
            // Return placeholder image or 404
            return $this->response->setStatusCode(404);
        }
        
        // Get file info
        $fileInfo = pathinfo($filePath);
        $mimeType = $this->getMimeType($fileInfo['extension'] ?? '');
        
        // Set headers
        $this->response->setHeader('Content-Type', $mimeType);
        $this->response->setHeader('Content-Length', filesize($filePath));
        $this->response->setHeader('Cache-Control', 'public, max-age=31536000'); // Cache for 1 year
        $this->response->setHeader('Expires', gmdate('D, d M Y H:i:s', time() + 31536000) . ' GMT');
        
        // Output file
        return $this->response->setBody(file_get_contents($filePath));
    }
    
    private function getMimeType($extension)
    {
        $mimeTypes = [
            'jpg' => 'image/jpeg',
            'jpeg' => 'image/jpeg',
            'png' => 'image/png',
            'gif' => 'image/gif',
            'webp' => 'image/webp',
            'svg' => 'image/svg+xml',
            'pdf' => 'application/pdf'
        ];
        
        return $mimeTypes[strtolower($extension)] ?? 'application/octet-stream';
    }
    
    public function placeholder($width = 300, $height = 200)
    {
        // Create a simple placeholder image
        $width = min(max((int)$width, 50), 1000);
        $height = min(max((int)$height, 50), 1000);
        
        // Create image
        $image = imagecreate($width, $height);
        
        // Colors
        $bgColor = imagecolorallocate($image, 240, 240, 240);
        $textColor = imagecolorallocate($image, 150, 150, 150);
        
        // Fill background
        imagefill($image, 0, 0, $bgColor);
        
        // Add text
        $text = $width . 'x' . $height;
        $fontSize = min($width, $height) / 10;
        $textX = ($width - strlen($text) * $fontSize * 0.6) / 2;
        $textY = ($height + $fontSize) / 2;
        
        imagestring($image, 3, $textX, $textY - 10, $text, $textColor);
        
        // Set headers
        $this->response->setHeader('Content-Type', 'image/png');
        $this->response->setHeader('Cache-Control', 'public, max-age=3600');
        
        // Output image
        ob_start();
        imagepng($image);
        $imageData = ob_get_contents();
        ob_end_clean();
        
        imagedestroy($image);
        
        return $this->response->setBody($imageData);
    }
}